<?php
/*
Plugin Name: Character Post Type with API and Upload
Description: Adds a Character post type with meta fields and allows fetch from API or manual upload.
Version: 1.3
Author: OSMZ
*/

if (!defined('ABSPATH')) exit;

/**
 * Register Post Type
 */
add_action('init', function(){
    register_post_type('character', array(
        'labels' => array(
            'name' => 'Characters',
            'singular_name' => 'Character',
            'menu_name' => 'Characters',
            'add_new' => 'Add New',
            'add_new_item' => 'Add New Character',
        ),
        'public' => true,
        'show_in_rest' => true,
        'menu_icon' => 'dashicons-id',
        'supports' => array('title'),
        'has_archive' => true,
        'rewrite' => array('slug' => 'characters'),
    ));
});

/**
 * Add Meta Box
 */
add_action('add_meta_boxes', function(){
    add_meta_box('cpt_character_details', 'Character Details', 'cpt_character_meta_box_callback', 'character');
});

/**
 * Meta Box HTML
 */
function cpt_character_meta_box_callback($post){
    wp_nonce_field('cpt_character_meta_nonce', 'cpt_character_meta_nonce_field');

    $character_id = get_post_meta($post->ID, '_character_id', true);
    $description = get_post_meta($post->ID, '_character_description', true);
    $nicknames = get_post_meta($post->ID, '_character_nicknames', true);
    $image_id = get_post_meta($post->ID, '_character_image_id', true);
    $image_url = $image_id ? wp_get_attachment_image_url($image_id, 'medium') : '';

    ?>
    <p>
        <label><strong>Character ID (MAL ID):</strong></label><br>
        <input type="number" id="cpt_character_id" name="cpt_character_id" value="<?php echo esc_attr($character_id); ?>" style="width:100%;">
        <button type="button" class="button" id="cpt_fetch_character">Fetch Character</button>
    </p>
    <hr>
    <p>
        <label><strong>Nicknames:</strong> (Comma-separated)</label><br>
        <input type="text" id="cpt_character_nicknames" name="cpt_character_nicknames" value="<?php echo esc_attr($nicknames); ?>" style="width:100%;">
    </p>
    <p>
        <label><strong>Description:</strong></label><br>
        <textarea id="cpt_character_description" name="cpt_character_description" rows="5" style="width:100%;"><?php echo esc_textarea($description); ?></textarea>
    </p>
    <p>
        <label><strong>Image (recommended size 225x335):</strong></label><br>
        <input type="hidden" id="cpt_character_image_id" name="cpt_character_image_id" value="<?php echo esc_attr($image_id); ?>">
        <button type="button" class="button" id="cpt_character_image_upload">Upload Image</button>
        <button type="button" class="button" id="cpt_character_image_remove" <?php echo $image_url ? '' : 'style="display:none;"'; ?>>Remove Image</button><br><br>
        <div id="cpt_character_image_preview">
            <?php if ($image_url): ?>
                <img src="<?php echo esc_url($image_url); ?>" style="max-width:150px;">
            <?php endif; ?>
        </div>
    </p>
    <?php
}

/**
 * Save Meta Data
 */
add_action('save_post', function($post_id){
    if (!isset($_POST['cpt_character_meta_nonce_field']) ||
        !wp_verify_nonce($_POST['cpt_character_meta_nonce_field'], 'cpt_character_meta_nonce')) {
        return;
    }

    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;
    if (!current_user_can('edit_post', $post_id)) return;

    update_post_meta($post_id, '_character_id', intval($_POST['cpt_character_id']));
    update_post_meta($post_id, '_character_description', wp_kses_post($_POST['cpt_character_description']));
    update_post_meta($post_id, '_character_nicknames', sanitize_text_field($_POST['cpt_character_nicknames']));
    update_post_meta($post_id, '_character_image_id', intval($_POST['cpt_character_image_id']));
});

/**
 * Admin Scripts for API Fetch + Upload
 */
add_action('admin_enqueue_scripts', function($hook){
    if ($hook !== 'post.php' && $hook !== 'post-new.php') return;

    wp_enqueue_script('jquery');
    wp_enqueue_media();

    wp_add_inline_script('jquery', <<<JS
jQuery(document).ready(function($){
    const postId = $('#post_ID').val();

    function uploadFromUrl(url, callback){
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: {
                action: 'cpt_upload_image',
                image_url: url,
                post_id: postId
            },
            success: function(res){
                if(res.success){
                    callback(res.data.id, res.data.url);
                } else {
                    alert('Failed to upload image');
                }
            }
        });
    }

    $('#cpt_fetch_character').on('click', function(e){
        e.preventDefault();
        const charId = $('#cpt_character_id').val();
        if(!charId){
            alert('Enter a character ID');
            return;
        }
        $(this).text('Fetching...');

        $.get('https://api.jikan.moe/v4/characters/'+charId+'/full', function(res){
            const data = res.data;
            $('#title').val(data.name);
            $('#cpt_character_description').val(data.about);
            $('#cpt_character_nicknames').val(data.nicknames.join(', '));

            const img = data.images.webp.image_url;
            uploadFromUrl(img, function(attachId, imgUrl){
                $('#cpt_character_image_id').val(attachId);
                $('#cpt_character_image_preview').html('<img src="'+imgUrl+'" style="max-width:150px;">');
                $('#cpt_character_image_remove').show();
            });

        }).fail(function(){
            alert('Failed to fetch character data');
        }).always(function(){
            $('#cpt_fetch_character').text('Fetch Character');
        });
    });

    $('#cpt_character_image_upload').on('click', function(e){
        e.preventDefault();
        const frame = wp.media({
            title: 'Select or Upload Image',
            button: { text: 'Use this image' },
            multiple: false
        });
        frame.on('select', function(){
            const attachment = frame.state().get('selection').first().toJSON();
            $('#cpt_character_image_id').val(attachment.id);
            $('#cpt_character_image_preview').html('<img src="'+attachment.url+'" style="max-width:150px;">');
            $('#cpt_character_image_remove').show();
        });
        frame.open();
    });

    $('#cpt_character_image_remove').on('click', function(e){
        e.preventDefault();
        $('#cpt_character_image_id').val('');
        $('#cpt_character_image_preview').html('');
        $(this).hide();
    });
});
JS
    );
});

/**
 * AJAX Upload Image from URL
 */
add_action('wp_ajax_cpt_upload_image', function(){
    if (!current_user_can('upload_files')) wp_send_json_error();

    $url = esc_url_raw($_POST['image_url']);
    $post_id = intval($_POST['post_id']);

    $tmp = download_url($url);
    if (is_wp_error($tmp)) wp_send_json_error();

    $file = array(
        'name'     => basename($url),
        'type'     => mime_content_type($tmp),
        'tmp_name' => $tmp,
        'error'    => 0,
        'size'     => filesize($tmp),
    );

    $overrides = array('test_form' => false);
    $results = wp_handle_sideload($file, $overrides);

    if (!empty($results['error'])) wp_send_json_error();

    $attachment = array(
        'post_mime_type' => $results['type'],
        'post_title'     => sanitize_file_name($results['file']),
        'post_content'   => '',
        'post_status'    => 'inherit',
    );

    $attach_id = wp_insert_attachment($attachment, $results['file'], $post_id);
    require_once(ABSPATH . 'wp-admin/includes/image.php');
    $attach_data = wp_generate_attachment_metadata($attach_id, $results['file']);
    wp_update_attachment_metadata($attach_id, $attach_data);

    wp_send_json_success(array(
        'id'  => $attach_id,
        'url' => wp_get_attachment_url($attach_id)
    ));
});


function mangosm_characters_support_enqueue_styles() {
    if (is_singular('character')) {
        wp_enqueue_style(
            'mangosm_characters_support-style',
            plugin_dir_url(__FILE__) . 'css/style.css'
        );
    }
}
add_action('wp_enqueue_scripts', 'mangosm_characters_support_enqueue_styles');


function enqueue_character_archive_styles() {
    if (is_post_type_archive('character')) {
        wp_enqueue_style(
            'character-archive-style',
            get_template_directory_uri() . '/css/character-archive.css',
            array(),
            '1.0'
        );
    }
}
add_action('wp_enqueue_scripts', 'enqueue_character_archive_styles');

